//  setupdlg.cpp  -  setup dialog sample code
//
//  This is a part of the MetaKit library.
//  Copyright (c) 1996 Meta Four Software.
//  All rights reserved.
/////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "catfish.h"
#include "setupdlg.h"
#include "pickdir.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CSetupDialog dialog

CSetupDialog::CSetupDialog(CWnd* pParent /*=NULL*/)
    : CDialog(CSetupDialog::IDD, pParent),
      m_exists (FALSE), m_timer (0), m_allowScan (FALSE)
{
    //{{AFX_DATA_INIT(CSetupDialog)
    m_name = "";
    //}}AFX_DATA_INIT
}

void CSetupDialog::DoDataExchange(CDataExchange* pDX)
{
    CDialog::DoDataExchange(pDX);
    //{{AFX_DATA_MAP(CSetupDialog)
    DDX_Control(pDX, IDOK, m_okBtn);
    DDX_Control(pDX, IDC_BROWSE_BTN, m_browseBtn);
    DDX_Control(pDX, IDC_DEL_BTN, m_deleteBtn);
    DDX_Control(pDX, IDC_ADD_BTN, m_addBtn);
    DDX_Control(pDX, IDC_UPDATE_BTN, m_updateBtn);
    DDX_Control(pDX, IDC_STATUS, m_status);
    DDX_Control(pDX, IDC_ROOT, m_root);
    DDX_Text(pDX, IDC_NAME, m_name);
    //}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CSetupDialog, CDialog)
    //{{AFX_MSG_MAP(CSetupDialog)
    ON_BN_CLICKED(IDC_ADD_BTN, OnAddBtn)
    ON_BN_CLICKED(IDC_UPDATE_BTN, OnUpdateBtn)
    ON_BN_CLICKED(IDC_DEL_BTN, OnDelBtn)
    ON_BN_CLICKED(IDC_BROWSE_BTN, OnBrowseBtn)
    ON_EN_CHANGE(IDC_NAME, OnChangeName)
    ON_EN_CHANGE(IDC_ROOT, OnChangeRoot)
    ON_WM_TIMER()
    ON_EN_KILLFOCUS(IDC_ROOT, OnKillfocusRoot)
    //}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CSetupDialog message handlers

    // this member is called by the ScanDisk code during its (lengthy) scan
bool CSetupDialog::UpdateStatus(const char* text)
{
    static DWORD lastTick = 0;
    
        // a null pointer forces immediate clear of the status text
    if (!text)
    {
        lastTick = 0;
        text = "";
    }
    
        // only refresh the status message every quarter second
    if (GetTickCount() > lastTick + 250)
    {
        lastTick = GetTickCount();
        m_status.SetWindowText(text);   
    }
    
    return cStatusHandler::UpdateStatus(text) && m_allowScan;
}

    // the name has changed, update other fields (either now, or a little later)
    // this delay is used to avoid excessive activity while a name is typed in
void CSetupDialog::NameChange(BOOL delay)
{
    CString s;
    if (m_name.IsEmpty())
        s = "(please enter name of catalog)";
    m_status.SetWindowText(s);

    m_exists = FALSE;
    
    if (delay)
    {       // arm the timer (if there is a catalog name)
        KillTimer(m_timer);
        m_timer = m_name.IsEmpty() ? 0 :
                    SetTimer(1, 500, 0); // timeout in 500 mS
    }
    else
        InspectCatalog();
}

    // scan through the directories, but be prepared to deal with premature abort
void CSetupDialog::ScanDirectories()
{
    CString s;
    m_root.GetWindowText(s);
    
        // prepare for a lengthy scan with a cancel option
    AdjustButtons(TRUE);                        
                            
    m_okBtn.EnableWindow(FALSE);
    m_browseBtn.SetWindowText("Abort");
    SetDefID(IDC_BROWSE_BTN);

    m_allowScan = TRUE;
                            
        // build a catalog of the specified directory tree
    c4_View dirs = fScanDirectories(s, this);
                            
        // restore normal situation
    m_allowScan = TRUE;
    m_okBtn.EnableWindow(TRUE);
    m_browseBtn.SetWindowText("Browse");
                            
    m_status.SetWindowText("Saving catalog...");
        // saving may take a litle time
    HCURSOR oldCursor = SetCursor(LoadCursor(0, IDC_WAIT));
    
    if (dirs.GetSize() > 0)
    {                        
        s = m_name + FILE_TYPE;
                                
            // clumsy, remove file to create smallest file
        CFileStatus fs;
        if (CFile::GetStatus(s, fs))
            CFile::Remove(s);
                                
            // this stores the catalog on file  
        c4_Storage storage (s, true);
        storage.Store("dirs", dirs);
        storage.Commit();
    }
                            
    SetCursor(oldCursor);
    
    InspectCatalog();
}     

    // go see if the catalog exists and get its file date and root path
void CSetupDialog::InspectCatalog()
{
    UpdateData();
                            
    CString s = GetCatalogDate(m_name);
    if (!s.IsEmpty())
        s = "Last change:  " + s;

    UpdateData(FALSE); // adjusts the name
                                                            
    m_status.SetWindowText(s);
    m_exists = !s.IsEmpty();
                                
    if (m_exists)
    {
            // load the root path name from the catalog
            // this is quick, due to on-demand loading
        c4_Storage storage (m_name + FILE_TYPE, false);
        c4_View dirs = storage.View("dirs");
        m_origRoot = pName (dirs[0]);
        m_root.SetWindowText(m_origRoot);
    }
}

    // adjust the button dimming and titles to reflect the current situation
void CSetupDialog::AdjustButtons(BOOL inScan_)
{
    CString s;
    m_root.GetWindowText(s);
    BOOL valid = !s.IsEmpty() && !inScan_;
    
        // don't enable buttons while the timer is running  
    m_addBtn.EnableWindow(valid && !m_exists && !m_timer && !m_name.IsEmpty());
    
    ASSERT(!(m_exists && m_timer)); // if it exists, timer cannot be running
    m_updateBtn.EnableWindow(valid && m_exists);
    m_deleteBtn.EnableWindow(valid && m_exists);
    
        // The default button is "OK" for existing files
        // If the root has been set for a new file, the default is "Add"
        // For new files, or if the root is different, the default is "Browse"
        // Never make "Change" the default, since that is a dangerous change
    SetDefID(valid &&  m_exists && s == m_origRoot ? IDOK :
             valid && !m_exists && !m_timer        ? IDC_ADD_BTN : 
                                                     IDC_BROWSE_BTN);
    
    if (!GetFocus())
        m_okBtn.SetFocus();
}
                            
        // called shortly after a change to the name or root edit control       
void CSetupDialog::OnTimer(UINT nIDEvent)
{
    KillTimer(m_timer);
    m_timer = 0;
    
    InspectCatalog();
    OnChangeRoot();
    
    CDialog::OnTimer(nIDEvent);
}
            
BOOL CSetupDialog::OnInitDialog()
{
    CenterWindow();
    CDialog::OnInitDialog();
    
    m_nameEditCtrl.SubclassDlgItem(IDC_NAME, this);
    
    NameChange(FALSE);
    AdjustButtons();
    
    return TRUE;  // return TRUE  unless you set the focus to a control
}

void CSetupDialog::OnAddBtn()
{
    OnKillfocusRoot();
    ScanDirectories();
    AdjustButtons();
}

void CSetupDialog::OnUpdateBtn()
{
    OnKillfocusRoot();
    ScanDirectories();
    AdjustButtons();
}

void CSetupDialog::OnDelBtn()
{
    if (AfxMessageBox("Do you want to permanently delete "
                        "the catalog named '" + m_name + "' ?",
                        MB_YESNO | MB_ICONEXCLAMATION | MB_DEFBUTTON2) == IDYES)
    {
        CFile::Remove(m_name + FILE_TYPE);

        NameChange(TRUE);
        AdjustButtons();
    }
}

void CSetupDialog::OnBrowseBtn()
{
    if (m_allowScan)
    {
        m_allowScan = FALSE;
        return;
    }
    
    CString dir = PickDirectory(this);
    if (!dir.IsEmpty())
        m_root.SetWindowText(dir); // triggers OnChangeRoot()
}

void CSetupDialog::OnChangeName()
{
    VERIFY(UpdateData());

    NameChange(TRUE);
    AdjustButtons();
}

void CSetupDialog::OnChangeRoot()
{
    CString s;
    m_root.GetWindowText(s);
                            
    BOOL changed = m_exists && s != m_origRoot;
    m_updateBtn.SetWindowText(changed ? "&Change" : "&Update");
    
    AdjustButtons();
}

void CSetupDialog::OnKillfocusRoot()
{
    CString s;
    m_root.GetWindowText(s);
    
    if (!s.IsEmpty())
    {
        s.MakeUpper();
        if (s.Right(1) != "\\")
            s += "\\"; // avoid handing "C:" to _fullpath

        char buf [_MAX_PATH];
        if (_fullpath(buf, s + "nul", sizeof buf))
        {
            s = buf;
        
            CFileStatus fs;
            if (CFile::GetStatus(s, fs))
                m_root.SetWindowText(s.Left(s.GetLength() - 4));
        }
    }
}
